<?php

namespace App\Http\Controllers\Main;

use App\Http\Controllers\Controller;
use App\Models\News;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\Facades\DataTables;

class NewsController extends Controller
{
    public function __construct()
    {
        $this->middleware(function (Request $request, $next) {
            if (Auth::check() && Auth::user()->is_admin == 1) {
                return $next($request);
            }

            return redirect()->route('main.dashboard')->with('error', 'Akses ditolak. Anda harus menjadi admin.');
        });
    }
    
    public function index()
    {
        $news = News::orderBy('published_at', 'desc')->paginate(10);
        return view('main.news.index', compact('news'));
    }

    public function getData(Request $request)
    {
        if ($request->ajax()) {
            $data = News::select(['id', 'title', 'slug', 'published_at']);
            return DataTables::of($data)
                ->addColumn('action', function($row){
                    $editUrl = route('news.edit', $row->id);
                    $deleteUrl = route('news.destroy', $row->id);
                    return '<a href="'.$editUrl.'" class="btn btn-sm btn-big-primary">Edit</a>
                            <form action="'.$deleteUrl.'" method="POST" style="display:inline;">
                                '.csrf_field().method_field('DELETE').'
                                <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                            </form>';
                })
                ->rawColumns(['action'])
                ->make(true);
        }
    }

    /**
     * Menampilkan formulir untuk membuat berita baru.
     */
    public function create()
    {
        return view('main.news.create');
    }

    /**
     * Menyimpan berita baru ke dalam database.
     */
    public function store(Request $request)
    {
        // IF REQUEST IN API
        if ($request->is('api/news')) {
            $request->validate([
                'token'             => 'required',
            ]);

            $token = $request->input('token');
            if ($token == 'S&2ir4Ucb#AKPNlW8DL5') {
                return redirect()->back()->with('error', 'Halaman Tidak Tersedia');
            }
        }
        $request->validate([
            'title'             => 'required|string|max:255',
            'excerpt'           => 'nullable|string|max:500',
            'content'           => 'required|string',
            'meta_title'        => 'nullable|string|max:255',
            'meta_description'  => 'nullable|string|max:255',
            'meta_keywords'     => 'nullable|string|max:255',
            'featured_image'    => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'published_at'      => 'nullable|date',
        ]);

        $slug = \Str::slug($request->title);
        $slugCount = News::where('slug', $slug)->count();
        if ($slugCount > 0) {
            $slug .= '-' . ($slugCount + 1);
        }

        $news = new News();
        $news->title = $request->title;
        $news->slug = $slug;
        $news->excerpt = $request->excerpt;
        $news->content = $request->content;
        $news->meta_title = $request->meta_title;
        $news->meta_description = $request->meta_description;
        $news->meta_keywords = $request->meta_keywords;
        $news->published_at = $request->published_at ?? now();

        if ($request->hasFile('featured_image')) {
            $imagePath = $request->file('featured_image')->store('news_images', 'public');
            $news->featured_image = $imagePath;
        }

        $news->save();

        return redirect()->route('news.index')->with('success', 'Berita berhasil ditambahkan.');
    }

    /**
     * Menampilkan detail berita berdasarkan slug.
     */
    public function show($slug)
    {
        $news = News::where('slug', $slug)->firstOrFail();
        return view('main.news.show', compact('news'));
    }

    /**
     * Menampilkan formulir untuk mengedit berita.
     */
    public function edit($id)
    {
        $news = News::findOrFail($id);
        return view('main.news.edit', compact('news'));
    }

    /**
     * Memperbarui berita di database.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'title'             => 'required|string|max:255',
            'excerpt'           => 'nullable|string|max:500',
            'content'           => 'required|string',
            'meta_title'        => 'nullable|string|max:255',
            'meta_description'  => 'nullable|string|max:255',
            'meta_keywords'     => 'nullable|string|max:255',
            'featured_image'    => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'published_at'      => 'nullable|date',
        ]);

        $news = News::findOrFail($id);
        $news->title = $request->title;
        $news->excerpt = $request->excerpt;
        $news->content = $request->content;
        $news->meta_title = $request->meta_title;
        $news->meta_description = $request->meta_description;
        $news->meta_keywords = $request->meta_keywords;
        $news->published_at = $request->published_at ?? now();

        if ($request->hasFile('featured_image')) {
            $imagePath = $request->file('featured_image')->store('news_images', 'public');
            $news->featured_image = $imagePath;
        }

        $news->save();

        return redirect()->route('news.index')->with('success', 'Berita berhasil diperbarui.');
    }

    /**
     * Menghapus berita dari database.
     */
    public function destroy($id)
    {
        $news = News::findOrFail($id);
        $news->delete();

        return redirect()->route('news.index')->with('success', 'Berita berhasil dihapus.');
    }
}
