<?php

namespace App\Http\Controllers\Main;

use App\Http\Controllers\Controller;
use App\Models\CategoryProduct;
use App\Models\News;
use App\Models\Product;
use App\Models\Service;
use App\Models\Transaction;
use App\Models\User;
use App\SmmPanelIndoTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Midtrans\Config;
use Midtrans\Snap;

class MainController extends Controller
{
    use SmmPanelIndoTrait;
    
    public function __construct()
    {
        // Konfigurasi Midtrans
        Config::$serverKey = config('services.midtrans.server_key');
        Config::$clientKey = config('services.midtrans.client_key');
        Config::$isProduction = config('services.midtrans.is_production');
        Config::$isSanitized = true;
        Config::$is3ds = true;
    }
    
    public function dashboard()
    {
        $categoryProduct = CategoryProduct::where('status', 1)->get();
        $products = Product::where('status', 1)->get();
        $total_order = Transaction::where('user_id', Auth::user()->id)->sum('charge');
        $total_transaction = Transaction::where('user_id', Auth::user()->id)->count();

        if (Auth::user()->is_admin == 1) {
            $user = User::orderBy('id', 'desc')->get();
            $transaction_today = Transaction::whereDate('created_at', now())->count();
            $balance = round($this->balance()['balance'], 2);
            $news = News::count();

            return view('main.dashboard', compact('categoryProduct', 'products', 'total_order', 'total_transaction', 'user', 'transaction_today', 'balance', 'news'));
        }
        
        return view('main.dashboard', compact('categoryProduct', 'products', 'total_order', 'total_transaction'));
    }

    public function order(Request $request)
    {
        $data = $request->all();
        $product = Product::find($data['product_id']);
        $service = Service::find($product->service_id);
        $total_order = (int)$data['quantity'] * (int)$product->rate;
        $total_order_with_tax = $total_order + ($total_order * 0.11);

        $user = Auth::user();

        // Data transaksi untuk Midtrans
        $transactionDetails = [
            'order_id' => '#INV' . time(), // Generate order ID unik
            'gross_amount' => $total_order_with_tax, // Harga paket
        ];

        $customerDetails = [
            'first_name' => $user->name,
            'email' => $user->email,
        ];

        $itemDetails = [
            [
                'id' => $product->id,
                'price' => $product->rate,
                'quantity' => $data['quantity'],
                'name' => $product->name,
            ],
        ];

        $snapToken = Snap::getSnapToken([
            'transaction_details' => $transactionDetails,
            'customer_details' => $customerDetails,
            'item_details' => $itemDetails,
        ]);

        Transaction::create([
            'user_id' => $user->id,
            'category_product_id' => $data['category_product_id'],
            'product_id' => $data['product_id'],
            'service_id' => $product->service_id,
            'order_id' => 0,
            'order_id_midtrans' =>$transactionDetails['order_id'],
            'link' => $data['link'],
            'quantity' => $data['quantity'],
            'charge' => $data['quantity'] * $product->rate,
            'remains' => 0,
            'currency' => 'IDR',
            'start_count' => 0,
            'status' => 'Need Payment',
            'snap_token' => $snapToken,
        ]);

        return response()->json([
            'status' => true,
            'snap_token' => $snapToken,
            'total_order' => $total_order,
            'total_order_with_tax' => $total_order_with_tax,
        ]);
    }

    public function webhook(Request $request)
    {
        $notification = $request->all();

        $orderIdMidtrans = $notification['order_id'];
        $transactionStatus = $notification['transaction_status'];

        $transaction = Transaction::where('order_id_midtrans', $orderIdMidtrans)->first();

        if ($transaction) {
            if ($transactionStatus == 'capture' || $transactionStatus == 'settlement') {
                $response = $this->createOrder($transaction->service_id, $transaction->link, $transaction->quantity);
                $transaction->order_id = $response['order'];
                
                $responseOrderStatus = $this->orderStatus($response['order']);
                $transaction->remains = $responseOrderStatus['remains'];
                $transaction->status = $responseOrderStatus['status'];
                $transaction->start_count = $responseOrderStatus['start_count'] ?? 0;
                $transaction->currency = $responseOrderStatus['currency'] ?? 'IDR';

                $transaction->status = 'Pending';
            } elseif ($transactionStatus == 'cancel' || $transactionStatus == 'deny') {
                $transaction->status = 'Canceled';
            } elseif ($transactionStatus == 'expire') {
                $transaction->status = 'Expired';
            }

            $transaction->save();
        }

        return response()->json(['status' => 'success']);
    }

    public function historyOrder()
    {
        $transactionPending = Transaction::where('user_id', Auth::user()->id)
                                        ->whereNotIn('status', ['Need Payment', 'Completed', 'Canceled', 'Expired'])
                                        ->get();

        foreach ($transactionPending as $key => $value) {
            $responseOrderStatus = $this->orderStatus($value->order_id);
            $value->remains = $responseOrderStatus['remains'];
            $value->status = $responseOrderStatus['status'];
            $value->start_count = $responseOrderStatus['start_count'];
            $value->currency = $responseOrderStatus['currency'];
            $value->save();
        }
        
        return view('main.history_order');
    }

    public function historyOrderData(Request $request)
    {
        $columns = [
            'id',
            'user_id',
            'category_product_id',
            'product_id',
            'service_id',
            'order_id',
            'order_id_midtrans',
            'link',
            'quantity',
            'charge',
            'remains',
            'currency',
            'start_count',
            'status',
            'snap_token',
            'created_at',
        ];

        $data = Transaction::orderBy('id', 'desc')
                            ->select($columns)
                            ->where('user_id', Auth::user()->id);

        return datatables()->of($data)
            ->addIndexColumn()
            ->addColumn('order_id', function ($row) {
                return $row->order_id;
            })
            ->addColumn('created_at', function ($row) {
                return $row->created_at->format('d-m-Y H:i:s');
            })
            ->addColumn('link', function ($row) {
                return "<a href='".$row->link."' target='_blank'>".$row->link."</a>";
            })
            ->addColumn('charge', function ($row) {
                return number_format($row->charge);
            })
            ->addColumn('start_count', function ($row) {
                return $row->start_count;
            })
            ->addColumn('quantity', function ($row) {
                return $row->quantity;
            })
            ->addColumn('layanan', function ($row) {
                return $row->product->name ?? '';
            })
            ->addColumn('status', function ($row) {
                return $row->status;
            })
            ->addColumn('remains', function ($row) {
                return $row->remains;
            })
            ->addColumn('action', function ($row) {
                $btn = '';
                if ($row->status == 'Need Payment') {
                    $btn .= '<a href="javascript:void(0)" class="btn btn-big-primary" onclick="payment(\''. $row->snap_token .'\')">';
                    $btn .= '<i class="fa fa-credit-card fa-fw"></i> Bayar';
                    $btn .= '</a>';
                }
                return $btn;
            })
            ->escapeColumns([])
            ->make(true);
    }

    public function orderGuide()
    {
        return view('main.order_guide');
    }

    public function tickets()
    {
        return view('main.tickets');
    }

    public function terms()
    {
        return view('main.terms');
    }

    public function contactMe()
    {
        return view('main.contact_us');
    }

    public function todoAccount()
    {
        return view('main.account');
    }

    public function account(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => 'required|string|max:255',
            'current_password' => 'required_with:password',
            'password' => 'nullable|min:8|confirmed',
            'password_confirmation' => 'required_with:password'
        ]);

        // Check current password if trying to change password
        if ($request->password && !Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Password saat ini tidak sesuai']);
        }

        // Update user data
        $user->name = $request->name;
        
        // Update password if provided
        if ($request->password) {
            $user->password = Hash::make($request->password);
        }

        $user->save();

        return back()->with('success', 'Akun berhasil diperbarui');
    }
}
